/*****************************************************************************
 * prodos/prodos_fs.h
 * Includes, definitions, and helpers pertaining to the ProDOS filesystem
 * implementation.
 *
 * Apple II ProDOS Filesystem Driver for Linux 2.4.x
 * Copyright (c) 2001 Matt Jensen.
 * This program is free software distributed under the terms of the GPL.
 *
 * 18-May-2001: Created
 *****************************************************************************/
#ifndef __PRODOS_PRODOS_FS_H
#define __PRODOS_PRODOS_FS_H

/*= Preprocessor Constants ==================================================*/

/* Block size. */
#define PRODOS_BLOCK_SIZE			0x200
#define PRODOS_BLOCK_SIZE_BITS		9 /* log base-2 of PRODOS_BLOCK_SIZE */

/* Number of entries per directory block. */
#define PRODOS_DIR_ENTS_PER_BLOCK	13

/* Various limits and maximums. */
#define PRODOS_MAX_NAME_LEN			0x0f
#define PRODOS_MAX_FILE_SIZE		0x00ffffff

/* ProDOS storage type values. */
#define PRODOS_STYPE_DELETED		0x00
#define PRODOS_STYPE_SEEDLING		0x10
#define PRODOS_STYPE_SAPLING		0x20
#define PRODOS_STYPE_TREE			0x30
#define PRODOS_STYPE_EXTENDED		0x50
#define PRODOS_STYPE_DIRECTORY		0xd0
#define PRODOS_STYPE_DIRHEADER		0xe0
#define PRODOS_STYPE_VOLHEADER		0xf0

/* ProDOS filetypes signifying simple text files. */
#define PRODOS_TEXT_TYPES			((char[]){0x04,0x2a,0xb0,0})

/* ProDOS access flags bits. */
#define PRODOS_ACCESS_DESTROY		0x80
#define PRODOS_ACCESS_RENAME		0x40
#define PRODOS_ACCESS_BACKUP		0x20
#define PRODOS_ACCESS_EXECUTE		0x08 /* not a standard access flag! */
#define PRODOS_ACCESS_HIDDEN		0x04
#define PRODOS_ACCESS_WRITE			0x02
#define PRODOS_ACCESS_READ			0x01

/* Positions of special blocks. */
#define PRODOS_HFS_DRIVER_BLOCK		0x00
#define PRODOS_PART_MAP_BLOCK		0x01
#define PRODOS_VOLUME_DIR_BLOCK		0x02

/* Length (in blocks) of the volume directory. Note that this MAY cause
 problems with disks that have been hacked to shorten or lengthen their volume
 directories. */
#define PRODOS_VOLUME_DIR_LENGTH	0x04

/* Various magic numbers. */
#define PRODOS_HFS_DRIVER_MAGIC		"ER"
#define PRODOS_PART_MAP_MAGIC_V1	"TS"
#define PRODOS_PART_MAP_MAGIC_V2	"PM"
#define PRODOS_IMAGE_2IMG_MAGIC		"2IMG"

/* Partition type tags. */
#define PRODOS_PART_TYPE_MAP		"Apple_Partition_Map"
#define PRODOS_PART_TYPE_HFS		"Apple_HFS"
#define PRODOS_PART_TYPE_PRODOS		"Apple_PRODOS"
#define PRODOS_PART_TYPE_FREE		"Apple_FREE"

/*= Preprocessor Macros =====================================================*/

/* ProDOS storage type/name length byte access macros. */
#define PRODOS_GET_STYPE(stnl) ((stnl[0])&0xf0)
#define PRODOS_GET_NLEN(stnl) ((stnl[0])&0x0f)
#define PRODOS_MAKE_STYPE_NLEN(st,nl) (((st)&0xf0)|((nl)&0x0f))

/* ProDOS indirect block entry access macros. */
#define PRODOS_GET_INDIRECT_ENTRY(pib,idx) ((u16)(((struct prodos_ind_block*)(pib))->msbyte[(idx)]<<8|((struct prodos_ind_block*)(pib))->lsbyte[(idx)]))
#define PRODOS_SET_INDIRECT_ENTRY(pib,idx,val) do{((struct prodos_ind_block*)(pib))->msbyte[(idx)]=(val)>>8&0xff;((struct prodos_ind_block*)(pib))->lsbyte[(idx)]=val&0xff;}while(0)

/*= Structure Definitions ===================================================*/

/* ProDOS directory block header. */
struct prodos_dir_block_header {
	u16 prev;
	u16 next;
} __attribute__((packed));

/* ProDOS volume directory header (follows directory block header.) */
struct prodos_volume_dir_header {
	u8 name[16]; /* name[0] is storage type & name length. */
	u8 reserved[6];
	u16 case_bits;
	u16 create_date;
	u16 create_time;
	u8 version;
	u8 min_version;
	u8 access;
	u8 entry_len;
	u8 entries_per_block;
	u16 entry_count;
	u16 bitmap_block;
	u16 block_count;
} __attribute__((packed));

/* ProDOS directory header (follows directory block header.) */
struct prodos_dir_header {
	u8 name[16]; /* name[0] is storage type & name length. */
	u8 reserved[8];
	u16 create_date;
	u16 create_time;
	u16 case_bits;
	u8 access;
	u8 entry_len;
	u8 entries_per_block;
	u16 entry_count;
	u16 parent_block;
	u8 parent_entry;
	u8 parent_entry_len;
} __attribute__((packed));

/* ProDOS directory entry. */
struct prodos_dir_entry {
	u8 name[16]; /* name[0] is storage type & name length. */
	u8 file_type;
	u16 key_block;
	u16 blocks_used;
	u8 eof[3];
	u16 create_date;
	u16 create_time;
	u16 case_bits;
	u8 access;
	u16 aux_type;
	u16 mod_date;
	u16 mod_time;
	u16 dir_block;
} __attribute__((packed));

/* First block of any ProDOS directory. */
struct prodos_dir_block_first {
	struct prodos_dir_block_header header;
	union {
		struct prodos_volume_dir_header vol_header;
		struct prodos_dir_header dir_header;
	} u;
	struct prodos_dir_entry entries[PRODOS_DIR_ENTS_PER_BLOCK-1];
} __attribute__((packed));

/* Second and subsequent blocks of any ProDOS directory. */
struct prodos_dir_block {
	struct prodos_dir_block_header header;
	struct prodos_dir_entry entries[PRODOS_DIR_ENTS_PER_BLOCK];
} __attribute__((packed));

/* ProDOS fork descriptor entry. */
struct prodos_fork_entry {
	u8 stype;
	u16 key_block;
	u16 blocks_used;
	u8 eof[3];
} __attribute__((packed));

/* ProDOS extended file (storage type 0x05) key block. */
struct prodos_ext_key_block {
	struct prodos_fork_entry data;
	u8 reserved1[248];
	struct prodos_fork_entry res;
	u8 reserved2[248];
} __attribute__((packed));

/* ProDOS indirect block (first- or second-level.) */
struct prodos_ind_block {
	u8 lsbyte[256];
	u8 msbyte[256];
} __attribute__((packed));

/* Apple partition map V1. */
struct prodos_partition_map_v1 {
	u16 signature;
	struct {
		u32 start;
		u32 size;
		u32 type;
	} entries[42];
} __attribute__((packed));

/* Apple partition map V2. One per block. */
struct prodos_partition_map_v2 {
	u16 signature;
	u16 padding;
	u32 num_blocks;
	u32 start;
	u32 size;
	u8 name[32];
	u8 type[32];
	u8 reserved[494];
} __attribute__((packed));

/* 2IMG image file header. */
struct prodos_2img_header {
	u8 magic[4];
	u8 creator[4];
	u16 header_len;
	u16 version;
	u32 image_format;
	u32 flags;
	u32 num_blocks;
	u32 data_offset;
	u32 data_len;
	u32 cmnt_offset;
	u32 cmnt_len;
	u32 creator_offset;
	u32 creator_len;
	u32 spare[4];
} __attribute__((packed));

/*= Helper Functions ========================================================*/

/* Translate a GS/OS time/date value into a Unix timestamp value. */
extern struct timezone sys_tz;
static inline time_t prodos_get_time(u16 date,u16 time) {
	int moff[] = {0,31,59,90,120,151,181,212,243,273,304,334};
	int year = date >> 9 & 0x7f;
	int month = (date >> 5 & 0x0f) - 1;
	int day = (date & 0x1f) - 1;
	int hour = time >> 8 & 0xff;
	int minute = time & 0xff;
	int yday = 0;
	
	/* Convert year value to proper format. */
	if (year < 40) year += 100;
	else if (year > 99) return 0;
	if (year < 70) return 0;

	/* Check month bounds and get yday. */
	if (month > 11) return 0;
	yday = moff[month] + day;

	/* Calculate time_t. GNU/Linux uses the standard Unix "seconds since
	 January 1, 1970 UTC" time format, but ProDOS ignores time zones (times
	 are stored in "wall clock" format.) Because of this, an attempt is made
	 to convert the value to local time. The resultant time value may or may
	 not match the original ProDOS timestamp since sys_tz doesn't always
	 contain accurate local time zone information and it doesn't distinguish
	 between standard and daylight savings time. Only the HOUR and AM/PM
	 portions of the time could be incorrect in any event. */
	return minute*60 + hour*3600 + yday*86400 + (year - 70)*31536000 +
	 ((year-69)/4)*86400 + sys_tz.tz_minuteswest * 60;
}

/* Translate ProDOS file access flags byte to umode_t. */
static inline umode_t prodos_get_mode(u8 access,int dir) {
	return (access & PRODOS_ACCESS_READ ? S_IRUGO : 0) |
		(access & PRODOS_ACCESS_WRITE && access & PRODOS_ACCESS_RENAME &&
		 access & PRODOS_ACCESS_DESTROY ? S_IWUGO : 0) |
		(access & PRODOS_ACCESS_EXECUTE || dir ? S_IXUGO : 0);
}

/*= Externals ===============================================================*/

/* pmap.c */
extern int prodos_find_partition(struct super_block *);

#endif
